<?php

namespace wellpack\wepak;

/**
 * Classe d'interrogation du service REST Wepak. La classe fournis des méthodes qui permettent
 * d'interroger l'API du service.
 * 
 * @example
 *      <pre>
 *      // instanciation du service
 *      $service = new \wellpack\wepak\WepakService($myApiKey, $urlService);
 *      
 *      // Création d'une requête de volume 
 *      $request = new \wellpack\wepak\VolumeRequest();
 *      
 *      // Définir les filtres de la requête
 *      $request->setCode_postal(75014);
 *      $request->setDept(75);
 *      $request->setCivilite(\wellpack\wepak\ServiceRequest::CIVILITE_HOMME);
 *      $request->setAgemin(30);
 *      $request->setAgemax(45);
 *      
 *      // Envoyer la requête et récupérer une réponse \wellpack\wepak\VolumeResponse
 *      $response = $service->getVolume($request);
 *      
 *      if ($response->isValid()) {
 *          // Si la réponse et valide, faire quelque chose avec les données
 *      } else {
 *          // Sinon récupérer le message d'erreur
 *          $message = $response->getErrorMessage();
 *      }
 *      </pre>
 *
 */
class WepackService {
    
    private $apiKey;
    
    private $url;
    
    public function __construct($apiKey, $url) {
        $this->apiKey = $apiKey;
        $this->url = $url;
    }
    
    /**
     * Demande le volume correspondant aux valeurs des filtres de la requête.
     * @param VolumeGlobalRequest $request
     * @return \wellpack\wepak\VolumeGlobalResponse
     */
    public function getVolumeGlobal(VolumeGlobalRequest $request) {
    	return new VolumeGlobalResponse($this->doRequest($request));
    }

    /**
     * Demande de volume correspondant aux valeurs des filtres de la requête.
     * @param VolumeRequest $request Requête à effectuer
     * @return VolumeResponse La réponse du service
     */
    public function getVolume(VolumeRequest $request) {
        return new VolumeResponse($this->doRequest($request));
    }
    
    /**
     * Demande l’envoi d’une campagne correspondant aux valeurs des filtres
     * de la requête.
     * @param SendRequest $request
     * @return \wellpack\wepak\SendResponse
     */
    public function getSend(SendRequest $request) {
    	return new SendResponse($this->doRequest($request));
    }
    
    /**
     * Demande l’envoi d’un sms pour tester la campagne.
     * @param SendTestRequest $request
     * @return \wellpack\wepak\SendTestResponse
     */
    public function getSendTest(SendTestRequest $request) {
        return new SendTestResponse($this->doRequest($request));
    }
    
    /**
     * Demande la requête sql a exécuté pour envoyer la campagne sms.
     * @param LocalityGroupCalcRequest $request
     * @return \wellpack\wepak\LocalityGroupCalcResponse
     */
    public function getLocalityGroupCalc(LocalityGroupCalcRequest $request) {
        return new LocalityGroupCalcResponse($this->doRequest($request), [
            ServiceRequest::KEY_DATAOUT => $request->getDataout()
        ]);
    }
    
    /**
     * Demande la liste des groupes d’intérêt père/fils configurer sur le site web.
     * @param ListInterestGroupRequest $request
     * @return \wellpack\wepak\ListInterestGroupResponse
     */
    public function getListInterestGroup(ListInterestGroupRequest $request) {
    	return new ListInterestGroupResponse($this->doRequest($request));
    }
    
    /**
     * Demander la liste des ci/qualif associé aux groupes d’intérêts père/fils
     * configurés sur le site web.
     * @param GetInteretValueRequest $request
     * @return \wellpack\wepak\GetInteretValueResponse
     */
    public function getInteretValue(GetInteretValueRequest $request) {
        return new GetInteretValueResponse($this->doRequest($request));
    }
    
    /**
     * Demander la création d’un lien court.
     * @param CreateLienCourtRequest $request
     * @return \wellpack\wepak\CreateLienCourtResponse
     */
    public function getCreateLienCourt(CreateLienCourtRequest $request) {
    	return new CreateLienCourtResponse($this->doRequest($request));
    }
    
    /**
     * Demander la liste des liens courts
     * @param ListLienCourtRequest $request
     * @return \wellpack\wepak\ListLienCourtResponse
     */
    public function getListLienCourt(ListLienCourtRequest $request) {
    	return new ListLienCourtResponse($this->doRequest($request));
    }
    
    /**
     * Demander la suppression d'un lien court
     * @param SupprLienCourtRequest $request
     * @return \wellpack\wepak\SupprLienCourtResponse
     */
    public function getSupprLienCourt(SupprLienCourtRequest $request) {
    	return new SupprLienCourtResponse($this->doRequest($request));
    }
    
    /**
     * Demander la création d’un repoussoir temporaire.
     * @param CreateRepoussoirTempRequest $request
     * @return \wellpack\wepak\CreateRepoussoirTempResponse
     */
    public function getCreateRepoussoirTemp(CreateRepoussoirTempRequest $request) {
    	return new CreateRepoussoirTempResponse($this->doRequest($request));
    }
    
    /**
     * Demander la création d’un repoussoir temporaire.
     * @param UpdateRepoussoirTempPrefixeRequest $request
     * @return \wellpack\wepak\UpdateRepoussoirTempPrefixeResponse
     */
    public function getUpdateRepoussoirTempPrefixe(UpdateRepoussoirTempPrefixeRequest $request) {
    	return new UpdateRepoussoirTempPrefixeResponse($this->doRequest($request));
    }
    
    
    private function doRequest(ServiceRequest $request) {
        
        $body = 'api_key=' . $this->apiKey . "&" . 'data=' . json_encode($request);
        
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_POST, true);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $body);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_URL, $this->url);
        // debug
//         curl_setopt($curl, CURLOPT_VERBOSE, true);
//         curl_setopt($curl, CURLOPT_STDERR, $verbose = fopen('php://temp', 'rw+'));
        
        $result = curl_exec($curl);
        
        if ($result === false) {
            $errno = curl_errno($curl);
            if ($errno > 0) {
                $result = '{"' . ServiceResponse::KEY_ERROR  . '": "' . $errno . ': ' . curl_strerror($errno) . '"}';
            }
        }
        
        curl_close($curl);
        
        return $result;
    }
}

